/*
 * SOURCE:  video.c
 * PROJECT: EasyTeX
 *
 * PURPOSE: routines for direct video access
 *
 * UPDATES: 08/15/1991 - major rewrite
 *
 * (c)M.Schollmeyer
 */
#include "video.h"

struct VideoBase *VideoBase;


static struct VideoBase vb = { 0, -1, 80, 25, 0, 0, 0, 0, NULL
};

struct VideoBase *InitVideo( int vm ) {

    unsigned int adapter;
    unsigned int vsegment;
    unsigned int rows;
    unsigned int oldmode;
    unsigned int _far *vdata;
    BOOL ret = TRUE;

    if( vm != -1 ) {
        _asm {
            mov ah, 0fh
            int 10h
            xor ah, ah
            mov oldmode, ax
            cmp al, byte ptr vm
            je  setvmodeok

            mov ah, 0
            mov al, byte ptr vm
            int 10h
            mov ah, 0fh
            int 10h
            cmp al, byte ptr vm
            je  setvmodeok
        }
        ret = FALSE;
setvmodeok:
        vb.OldMode = oldmode;
    }

    FP_SEG( vdata ) = 0x40;
    FP_OFF( vdata ) = 0x49;
    vb.Mode = *(unsigned char _far *)vdata;

    FP_OFF( vdata ) = 0x4a;
    vb.ScreenWidth = *vdata;

    FP_OFF( vdata ) = 0x84;
    vb.ScreenHeight = *vdata + 1;

    FP_OFF( vdata ) = 0x4c;
    vb.BufferLen = *vdata;

    _asm {
        mov adapter, 0
        mov ah, 15
        int 10h
        cmp al, 7
        je  mono
        or  adapter, VIDEO_COLOR
        jmp testegavga
mono:
        or  adapter, VIDEO_MONO
        mov dx, 03bah
        xor bl, bl
        in  al, dx
        and al, 80h
        mov ah, al
        mov cx, 8000h
testherc:
        in  al, dx
        and al, 80h
        cmp al, ah
        je  loopherc
        inc bl
        cmp bl, 10
        jb  loopherc
        or  adapter, VIDEO_HERC
        mov vsegment, 0b000h
        jmp testegavga
loopherc:
        loop testherc
testegavga:
        mov bh, 0ffh
        mov cl, 0ffh
        mov ah, 12h
        mov bl, 10h
        int 10h

        cmp bh, 01h
        ja  noega
        cmp cl, 0fh
        ja  noega
        or  adapter, VIDEO_EGAVGA
        mov vsegment, 0b800h
noega:
        nop
    }

    vb.Adapter = adapter;
    vb.Segment = vb.BufferStart = vsegment;

    if( ret )
        return &vb;
    else
        return NULL;
}

void VideoSetPage( int page ) {

    vb.Segment = vb.BufferStart + page * (vb.BufferLen >> 4);
}


/*
 *    Name: VideoExc
 *  Return: void
 * Purpose: swaps a rectangle in video buffer with a specified buffer
 *
 *
 *
 * (c)M.Schollmeyer
 */
void VideoExc( buffer, ulx, uly, lrx, lry )
unsigned int _far *buffer, ulx, uly, lrx, lry;
{
    unsigned int far *vp;
    unsigned int i, j, save;

    FP_SEG( vp ) = vb.Segment;
    FP_OFF( vp ) = 2 * ( uly * SCREEN_WIDTH + ulx );

    for( i = lry - uly; i >= 0; --i )
    {
        for( j = lrx - ulx; j >= 0; --j )
        {
            save = *vp;
            *vp++ = *buffer;
            *buffer++ = save;
        }
        vp += SCREEN_WIDTH;
    }
}

/*
 *    Name: VideoFill
 *  Return: void
 * Purpose: Fills a rectangle with a specified color
 *
 *
 *
 * (c)M.Schollmeyer
 */
void VideoFill( color, ulx, uly, lrx, lry )
colorreg color;
unsigned int ulx, uly, lrx, lry;
{
    unsigned int far *vp;
    int i, j;

    FP_SEG( vp ) = vb.Segment;
    FP_OFF( vp ) = 2 * ( uly * SCREEN_WIDTH + ulx );

    for( i = (int)( lry - uly ); i >= 0; --i )
    {
        for( j = (int)( lrx - ulx ); j >= 0; --j )
        {
            *vp++ = vb.Palette[color] << 8;
        }
        vp += SCREEN_WIDTH - (int)( lrx - ulx + 1 );
    }
}

/*
 *    Name: VideoOut
 *  Return: void
 * Purpose: Puts a character to the video buffer
 *
 *
 *
 * (c)M.Schollmeyer
 */
void VideoOut( colorreg color, unsigned char c, unsigned int x, unsigned int y )
{
    unsigned int far *vp;

    FP_SEG( vp ) = vb.Segment;
    FP_OFF( vp ) = 2 * ( y * SCREEN_WIDTH + x );

    *vp = c|(vb.Palette[color]<<8);
}

/*
 *    Name: VideoStr
 *  Return: void
 * Purpose: Puts a string to the video buffer
 *
 *
 *
 * (c)M.Schollmeyer
 */
void VideoStr( colorreg color, unsigned char *cp, unsigned int x, unsigned int y )
{
    unsigned int far *vp;
    unsigned int attr = vb.Palette[color];

    FP_SEG( vp ) = vb.Segment;
    FP_OFF( vp ) = 2 * ( y * SCREEN_WIDTH + x );

    while( *cp )
        *vp++ = *cp++|(vb.Palette[color]<<8);
}

/*
 *    Name: VideoPut
 *  Return: void
 * Purpose: Puts a character to the video buffer
 *
 *
 *
 * (c)M.Schollmeyer
 */
void VideoPut( unsigned int c, unsigned int x, unsigned int y )
{
    unsigned int far *vp;

    FP_SEG( vp ) = vb.Segment;
    FP_OFF( vp ) = 2 * ( y * SCREEN_WIDTH + x );

    *vp = c;
}


/*
 *    Name: VideoAttr
 *  Return: void
 * Purpose: modifies the attributes of a character.
 *
 *
 *
 * (c)M.Schollmeyer
 */
void VideoAttr( colorreg color, unsigned int x, unsigned int y )
{
    unsigned int _far *vp;

    FP_SEG( vp ) = vb.Segment;
    FP_OFF( vp ) = 2 * ( y * SCREEN_WIDTH + x );

    *vp = ( *vp & 0xff ) | ( vb.Palette[color]<< 8 );
}


/*
 *    Name: VideoGet
 *  Return: the character read
 * Purpose: Gets a character from the video buffer
 *
 *
 *
 * (c)M.Schollmeyer
 */
unsigned int VideoGet( unsigned int x, unsigned int y )
{
    unsigned int ret;
    unsigned int _far *vp;

    FP_SEG( vp ) = vb.Segment;
    FP_OFF( vp ) = 2 * ( y * SCREEN_WIDTH + x );
    return *vp;
}

/*
 *    Name: VideoInv
 *  Return: void
 * Purpose: Inverts a rectangle in the video buffer
 *
 *
 *
 * (c)M.Schollmeyer
 */
void VideoInv( int ulx, int uly, int lrx, int lry, unsigned int flags )
{
    int x, y;

    for( x = ulx; x <= lrx; ++x )
    {
        for( y = uly; y <= lry; ++y )
        {
            VideoPut( VideoGet( x, y ) ^ flags, x, y );
        }
    }
}


/* end of file video.c */
