/* ln.pack.c -- link template for packed executables.

   Copyright (C) 1994, 1995 Ralph Schleicher  */

/* This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */


#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <io.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <process.h>
#include <ansidecl.h>
#include "ln.h"


static void EXFUN (catch, (int sig));
static void volatile EXFUN (die, (int code, int err));


static char link[LN_TEMP_LENGTH] = LN_TEMP_MAGIC;
static char temp[LN_TEMP_LENGTH];
static int handle = -1;
static char *prog;


int
DEFUN (main, (ac, av, ep),
int ac AND
char **av AND
char **ep)
{
  char *p;
  int c, s;

  /* How am I called.  */

  prog = av[0];

  /* Catch keyboard-generated interrupts if not running in the
     background.  */

  if (signal (SIGINT, SIG_IGN) != SIG_IGN)
    {
      if (signal (SIGINT, catch) == SIG_ERR)
	die (-1, errno);
      if (signal (SIGQUIT, catch) == SIG_ERR)
	die (-1, errno);
      if (signal (SIGBREAK, catch) == SIG_ERR)
	die (-1, errno);
    }

  /* Test if there is a special temporary directory or use the current
     working directory if not.  */

  p = getenv ("TMP");
  if (p)
    goto gotcha;
  p = getenv ("TMPDIR");
  if (p)
    goto gotcha;
  p = getenv ("TEMP");
  if (p)
    goto gotcha;
  p = ".";

 gotcha:

  /* Check if there is enough space in the buffer and create a
     temporary file name.  */

  if (strlen (p) + 1 + 8 + 4 + 1 > LN_TEMP_LENGTH)
    die (-1, ERANGE);

  strcpy (temp, p);
  p = strchr (temp, 0);
  strcpy (p, "\\lnXXXXXX");
  if (mktemp (++p) == 0)
    die (-1, errno);
  strcat (p, ".exe");

  /* Open the temporary file for writing.  */

  handle = open (temp, O_WRONLY | O_CREAT | O_TRUNC | O_BINARY, S_IWRITE);
  if (handle == -1)
    die (-1, errno);

  /* Save the standard output handle and redirect it to the previously
     created temporary file.  */

  s = dup (STDOUT_FILENO);
  if (s == -1 || dup2 (handle, STDOUT_FILENO) == -1)
    die (-1, errno);

  /* Call gzip(1) decompressing the executable to standard output
     (which is our temporary file, now).  */

  if (spawnlp (P_WAIT, "gzip", "gzip", "-cd", link, 0) == -1)
    die (-1, errno);

  /* Restore standard output and close the open file descriptors.  */

  if (dup2 (s, STDOUT_FILENO) == -1 || close (s) == -1 || close (handle) == -1)
    die (-1, errno);

  /* Finally, execute the decompressed program under my own name.  */

  c = spawnve (P_WAIT, temp, av, ep);
  if (c == -1)
    die (-1, errno);

  /* die() does the rest.  */

  die (c, 0);
}


static void
DEFUN (catch, (sig),
int sig)
{
  char *s = 0;

  signal (sig, SIG_ACK);
  signal (sig, SIG_IGN);

  switch (sig)
    {
    case SIGINT: s = "SIGINT"; break;
    case SIGQUIT: s = "SIGQUIT"; break;
    case SIGBREAK: s = "SIGBREAK"; break;
    }

  fprintf (stderr, "%s: Ouch -- received %s\n", prog, s);

  die (-1, 0);
}


static void volatile
DEFUN (die, (code, err),
int code AND
int err)
{
  if (handle != -1)
    {
      close (handle);
      remove (temp);
    }

  if (err)
    fprintf (stderr, "%s: %s\n", prog, sys_errlist[err]);

  exit (code);
}
